/* --------------------------------------------------------------
  CustomStylesModal.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals CodeMirror, -Modal */

'use strict';

import EventsEmitter from '../libs/EventsEmitter';
import Modal from '../libs/Modal';

/**
 * Custom Styles Modal Controller
 *
 * This controller handles the functionality of the "Custom Styles" modal dialog where the
 * user is able to write custom CSS or SASS that is going to be included in the template.
 */
export default class CustomStylesModal {
	/**
	 * Class Constructor 
	 * 
	 * @param {String} customStyles Contains the custom styles.
	 */
	constructor(customStyles) {
		StyleEdit.Validator.isString(customStyles);
		
		/**
		 * Modal Selector
		 *
		 * @type {jQuery}
		 */
		this.$modal = null;
		
		/**
		 * Custom Styles
		 *
		 * @type {String}
		 */
		this.customStyles = customStyles;
		
		/**
		 * CodeMirror Instance
		 *
		 * @type {CodeMirror}
		 */
		this.codeEditor = null;
	}
	
	/**
	 * Initialize Controller
	 */
	initialize() {
		const data = {
			title_custom_styles_modal: StyleEdit.Language.translate('title_custom_styles_modal', 'style_edit'),
			option_save: StyleEdit.Language.translate('option_save', 'style_edit'),
			option_apply: StyleEdit.Language.translate('option_apply', 'style_edit'),
			option_cancel: StyleEdit.Language.translate('option_cancel', 'style_edit'),
			custom_styles: this.customStyles
		};
		
		this.$modal = Modal.show($('#custom-styles-modal-template'), data);
		this.$modal.find('.nav-tabs li:first a').tab('show');
		this.codeEditor = CodeMirror.fromTextArea(this.$modal.find('.custom-styles-input').get(0), {
			value: this.customStyles,
			mode: 'text/x-sass',
			lineNumbers: true,
			tabSize: 4
		});
		
		this._bindEventHandlers();
		
		$.material.init();
		
		EventsEmitter.triggerControllerInitialized(this.$modal, ['CustomStylesModal']);
	}
	
	/**
	 * Destroy Controller
	 */
	destroy() {
		Modal.hide(this.$modal);
		EventsEmitter.triggerControllerDestroyed(this.$modal, ['CustomStylesModal']);
	}
	
	/**
	 * Event: Save Button Click
	 *
	 * @private
	 */
	_onClickSave() {
		EventsEmitter.triggerCustomStylesChanged(this.$modal, [this.codeEditor.getValue()]);
		this.destroy();
	}
	
	/**
	 * Event: Quick-Save Button Click
	 *
	 * @private
	 */
	_onClickQuickSave() {
		EventsEmitter.triggerCustomStylesChanged(this.$modal, [this.codeEditor.getValue()]);
		$('.btn.save', '.style-edit-container').trigger('click');
	}
	
	/**
	 * Event: Cancel Button Click
	 *
	 * @private
	 */
	_onClickCancel() {
		this.destroy();
	}
	
	/**
	 * Event: Cancel Button Click
	 *
	 * @private
	 */
	_onClickFullscreenToggle() {
		$('.custom-styles', this.$modal).toggleClass('fullscreen');
		$('.toggle-icon', this.$modal).toggleClass('hidden');
	}
	
	/**
	 * Bind Event Handlers
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$modal
			.on('click', '.btn.quicksave', () => this._onClickQuickSave())
			.on('click', '.btn.save', () => this._onClickSave())
			.on('click', '.btn.cancel', () => this._onClickCancel())
			.on('click', '.btn.fullscreen-toggle', () => this._onClickFullscreenToggle());
	}
}
